function isValidDataUrl(url) {
  return typeof url === 'string' && url.startsWith('data:');
}

chrome.runtime.onMessage.addListener(async (message, sender) => {
  // --- Existing Email Logic ---
  if (message.action === "openGmailWithImage") {
    if (!isValidDataUrl(message.imageData)) {
      console.error('Invalid image data received:', message.imageData);
      return;
    }

    try {
      const gmailComposeUrl = new URL('https://mail.google.com/mail/');
      gmailComposeUrl.searchParams.set('view', 'cm');
      gmailComposeUrl.searchParams.set('fs', '1');
      if (message.subject) { gmailComposeUrl.searchParams.set('su', message.subject); }
      if (message.recipient) { gmailComposeUrl.searchParams.set('to', message.recipient); }

      // Use chrome.windows.create() instead of chrome.tabs.create().
      const createdWindow = await chrome.windows.create({
        url: gmailComposeUrl.toString(),
        type: 'popup',
        width: 800, // Customize size
        height: 600, // Customize size
        left: 100, // Customize position
        top: 100 // Customize position
      });

      const newTabId = createdWindow.tabs[0].id;

      // Wait for the new tab to be fully loaded and the Gmail UI to be ready.
      await new Promise(resolve => {
        const listener = (tabId, changeInfo) => {
          if (tabId === newTabId && changeInfo.status === 'complete') {
            setTimeout(() => {
              chrome.tabs.onUpdated.removeListener(listener);
              resolve();
            }, 4000); // Wait 4 seconds to ensure UI is ready.
          }
        };
        chrome.tabs.onUpdated.addListener(listener);
      });

      // Inject the script to populate the email body.
      await chrome.scripting.executeScript({
        target: { tabId: newTabId },
         func: async (text, imageData) => {
          // This function is executed in the Gmail tab's context.

          let composeDiv = document.querySelector('div[contenteditable="true"]');
          if (!composeDiv) {
            // Wait for the compose div to appear.
            await new Promise(resolve => {
              const observer = new MutationObserver((mutations, obs) => {
                const foundDiv = document.querySelector('div[contenteditable="true"]');
                if (foundDiv) {
                  composeDiv = foundDiv; // Update composeDiv reference
                  obs.disconnect();
                  resolve();
                }
              });
              observer.observe(document.body, { childList: true, subtree: true });
            });
          }
          
          const b64toBlob = (b64Data, contentType = '', sliceSize = 512) => {
            const byteCharacters = atob(b64Data);
            const byteArrays = [];
            for (let offset = 0; offset < byteCharacters.length; offset += sliceSize) {
              const slice = byteCharacters.slice(offset, offset + sliceSize);
              const byteNumbers = new Array(slice.length);
              for (let i = 0; i < slice.length; i++) {
                byteNumbers[i] = slice.charCodeAt(i);
              }
              byteArrays.push(new Uint8Array(byteNumbers));
            }
            return new Blob(byteArrays, { type: contentType });
          };

          const mimeMatch = imageData.match(/:(.*?);/);
          const mimeType = mimeMatch ? mimeMatch[1] : 'application/octet-stream';
          const base64Data = imageData.split(',')[1];
          const imageBlob = b64toBlob(base64Data, mimeType);
          const fileName = `image.${mimeType.split('/')[1]}`;
          const imageFile = new File([imageBlob], fileName, { type: mimeType });

          const dataTransfer = new DataTransfer();
          dataTransfer.items.add(imageFile);

          const pasteEvent = new ClipboardEvent('paste', {
            clipboardData: dataTransfer,
            bubbles: true,
            cancelable: true
          });

          // Focus the compose area before pasting
          composeDiv.focus();
          composeDiv.dispatchEvent(pasteEvent);
          
          // Use a message passing approach instead of execCommand for text
          // to prevent potential interference with the paste event.
          setTimeout(() => {
              const range = document.createRange();
              const sel = window.getSelection();
              range.selectNodeContents(composeDiv);
              range.collapse(false); // Collapse to the end of the content.
              sel.removeAllRanges();
              sel.addRange(range);
              
              const textNode = document.createTextNode(`\n${text}`);
              range.insertNode(textNode);
          }, 1000); // Give Gmail time to process the paste.
        },
        args: [message.body, message.imageData]
      });
      
    } catch (error) {
      console.error('Error in background script (Email):', error);
    }
  } 
  
  // --- NEW Download Logic ---
  else if (message.action === "startImageDownload") {
    const { imageData, filename } = message;

//    if (!isValidDataUrl(imageData)) {
//      console.error('Invalid image data received for download:', imageData);
 //     return;
//    }

    try {
      // The Data URL (base64 image data) is directly used as the download source
      chrome.downloads.download({
        url: imageData, 
        filename: filename, // Suggested file name
        saveAs: false // Set to true if you want the "Save As" dialog to appear
      }, (downloadId) => {
        if (chrome.runtime.lastError) {
          console.error("Download failed:", chrome.runtime.lastError.message);
        } else {
          console.log(`Download started with ID: ${downloadId}`);
        }
      });
    } catch (error) {
      console.error('Error in background script (Download):', error);
    }
  }
});

